/*
 * Decompiled with CFR 0.152.
 */
package com.it_nomads.fluttersecurestorage;

import android.app.KeyguardManager;
import android.content.Context;
import android.content.SharedPreferences;
import android.hardware.biometrics.BiometricManager;
import android.hardware.biometrics.BiometricPrompt;
import android.os.Build;
import android.os.CancellationSignal;
import android.security.keystore.KeyGenParameterSpec;
import android.util.Base64;
import android.util.Log;
import androidx.annotation.NonNull;
import com.it_nomads.fluttersecurestorage.FlutterSecureStorageConfig;
import com.it_nomads.fluttersecurestorage.SecurePreferencesCallback;
import com.it_nomads.fluttersecurestorage.ciphers.KeyCipher;
import com.it_nomads.fluttersecurestorage.ciphers.StorageCipher;
import com.it_nomads.fluttersecurestorage.ciphers.StorageCipherFactory;
import com.it_nomads.fluttersecurestorage.crypto.EncryptedSharedPreferences;
import com.it_nomads.fluttersecurestorage.crypto.MasterKey;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;

public class FlutterSecureStorage {
    private static final String TAG = "FlutterSecureStorage";
    private static final Charset charset = StandardCharsets.UTF_8;
    private static final String SHARED_PREFERENCES_CONFIG_NAME = "FlutterSecureStorageConfiguration";
    private FlutterSecureStorageConfig config;
    @NonNull
    private final Context context;
    private SharedPreferences preferences;
    private StorageCipher storageCipher;
    private StorageCipherFactory storageCipherFactory;

    public FlutterSecureStorage(Context context) {
        this.context = context.getApplicationContext();
    }

    public String addPrefixToKey(String key) {
        return this.config.getSharedPreferencesKeyPrefix() + "_" + key;
    }

    public boolean containsKey(String key) {
        return this.preferences.contains(key);
    }

    public String read(String key) throws Exception {
        try {
            return this.readUnsafe(key);
        }
        catch (Exception e) {
            if (this.handleStorageError("read", key, e)) {
                return this.readUnsafe(key);
            }
            throw e;
        }
    }

    private String readUnsafe(String key) throws Exception {
        String rawValue = this.preferences.getString(key, null);
        if (this.config.isUseEncryptedSharedPreferences() && !this.config.shouldMigrateOnAlgorithmChange()) {
            return rawValue;
        }
        return this.decodeRawValue(rawValue);
    }

    public Map<String, String> readAll() throws Exception {
        try {
            return this.readAllUnsafe();
        }
        catch (Exception e) {
            if (this.handleStorageError("readAll", null, e)) {
                return this.readAllUnsafe();
            }
            throw e;
        }
    }

    private Map<String, String> readAllUnsafe() throws Exception {
        Map raw = this.preferences.getAll();
        HashMap<String, String> all = new HashMap<String, String>();
        for (Map.Entry entry : raw.entrySet()) {
            String keyWithPrefix = (String)entry.getKey();
            if (!keyWithPrefix.contains(this.config.getSharedPreferencesKeyPrefix())) continue;
            String key = ((String)entry.getKey()).replaceFirst(this.config.getSharedPreferencesKeyPrefix() + '_', "");
            if (this.config.isUseEncryptedSharedPreferences() && !this.config.shouldMigrateOnAlgorithmChange()) {
                all.put(key, (String)entry.getValue());
                continue;
            }
            String rawValue = (String)entry.getValue();
            String value = this.decodeRawValue(rawValue);
            all.put(key, value);
        }
        return all;
    }

    public void write(String key, String value) throws Exception {
        try {
            this.writeUnsafe(key, value);
        }
        catch (Exception e) {
            if (this.handleStorageError("write", key, e)) {
                this.writeUnsafe(key, value);
            }
            throw e;
        }
    }

    private void writeUnsafe(String key, String value) throws Exception {
        SharedPreferences.Editor editor = this.preferences.edit();
        if (this.config.isUseEncryptedSharedPreferences() && !this.config.shouldMigrateOnAlgorithmChange()) {
            editor.putString(key, value);
        } else {
            byte[] result = this.storageCipher.encrypt(value.getBytes(charset));
            editor.putString(key, Base64.encodeToString((byte[])result, (int)0));
        }
        editor.apply();
    }

    public void delete(String key) {
        SharedPreferences.Editor editor = this.preferences.edit();
        editor.remove(key);
        editor.apply();
    }

    public void deleteAll() {
        SharedPreferences.Editor editor = this.preferences.edit();
        editor.clear();
        editor.apply();
    }

    protected void initialize(FlutterSecureStorageConfig config, final SecurePreferencesCallback<Void> callback) {
        this.config = config;
        if (this.preferences != null) {
            callback.onSuccess(null);
            return;
        }
        final SharedPreferences nonEncryptedPreferences = this.context.getSharedPreferences(config.getSharedPreferencesName(), 0);
        final SharedPreferences configSource = this.context.getSharedPreferences(SHARED_PREFERENCES_CONFIG_NAME, 0);
        Boolean isAlreadyMigrated = this.getEncryptedPrefsMigrated(configSource);
        if (!isAlreadyMigrated.booleanValue()) {
            try {
                final SharedPreferences encryptedPreferences = this.initializeEncryptedSharedPreferencesManager(this.context);
                if (this.hasDataInEncryptedSharedPreferences(encryptedPreferences)) {
                    Log.w((String)TAG, (String)"Found data in EncryptedSharedPreferences (deprecated)");
                    Log.w((String)TAG, (String)"EncryptedSharedPreferences is DEPRECATED and will be removed in a later version");
                    Log.w((String)TAG, (String)"The Jetpack Security library has been deprecated by Google.");
                    if (!config.shouldMigrateOnAlgorithmChange()) {
                        Log.w((String)TAG, (String)"Data found in EncryptedSharedPreferences, but migrateOnAlgorithmChange is set to false.");
                        Log.w((String)TAG, (String)"Set migrateOnAlgorithmChange=true to migrate to custom cipher storage.");
                        if (config.isUseEncryptedSharedPreferences()) {
                            Log.i((String)TAG, (String)"Using EncryptedSharedPreferences (migration disabled).");
                            this.preferences = encryptedPreferences;
                            callback.onSuccess(null);
                            return;
                        }
                        Log.e((String)TAG, (String)"Data exists in EncryptedSharedPreferences but encryptedSharedPreferences=false and migrateOnAlgorithmChange=false.");
                        Log.e((String)TAG, (String)"Either set encryptedSharedPreferences=true to use the old data, or set migrateOnAlgorithmChange=true to migrate it.");
                        callback.onError(new Exception("EncryptedSharedPreferences data found but migration is disabled. Set migrateOnAlgorithmChange=true to migrate."));
                        return;
                    }
                    Log.i((String)TAG, (String)"Migrating data from EncryptedSharedPreferences to custom cipher storage...");
                    if (config.isUseEncryptedSharedPreferences()) {
                        Log.w((String)TAG, (String)"Your data will be automatically migrated. You can safely remove encryptedSharedPreferences from your config after migration.");
                    }
                    Log.i((String)TAG, (String)"Migrating data from EncryptedSharedPreferences to selected custom cipher storage...");
                    this.initializeStorageCipher(configSource, new SecurePreferencesCallback<Void>(){

                        @Override
                        public void onSuccess(Void unused) {
                            try {
                                FlutterSecureStorage.this.migrateFromEncryptedSharedPreferences(encryptedPreferences, nonEncryptedPreferences);
                                FlutterSecureStorage.this.preferences = nonEncryptedPreferences;
                                Log.i((String)FlutterSecureStorage.TAG, (String)"Migration completed successfully. Now using custom cipher storage.");
                                FlutterSecureStorage.this.setEncryptedPrefsMigrated(configSource);
                                callback.onSuccess(null);
                            }
                            catch (Exception e) {
                                Log.e((String)FlutterSecureStorage.TAG, (String)"Migration failed. Falling back to EncryptedSharedPreferences.", (Throwable)e);
                                FlutterSecureStorage.this.preferences = encryptedPreferences;
                                callback.onSuccess(null);
                            }
                        }

                        @Override
                        public void onError(Exception e) {
                            Log.e((String)FlutterSecureStorage.TAG, (String)"Cipher initialization failed during migration. Using EncryptedSharedPreferences.", (Throwable)e);
                            FlutterSecureStorage.this.preferences = encryptedPreferences;
                            callback.onSuccess(null);
                        }
                    });
                    return;
                }
                Log.d((String)TAG, (String)"No data found in EncryptedSharedPreferences.");
                if (config.isUseEncryptedSharedPreferences() && !config.shouldMigrateOnAlgorithmChange()) {
                    Log.w((String)TAG, (String)"Using EncryptedSharedPreferences (deprecated). Consider migrating to custom ciphers.");
                    this.preferences = encryptedPreferences;
                    callback.onSuccess(null);
                    return;
                }
            }
            catch (Exception e) {
                Log.e((String)TAG, (String)"EncryptedSharedPreferences initialization failed. Falling back to custom ciphers.", (Throwable)e);
            }
        }
        if (this.preferences == null) {
            if (config.isUseEncryptedSharedPreferences() && isAlreadyMigrated.booleanValue()) {
                Log.i((String)TAG, (String)"Data already migrated, encryptedSharedPreferences ignored and can be safely removed.");
            }
            this.preferences = nonEncryptedPreferences;
            this.initializeStorageCipher(configSource, callback);
        }
    }

    private void initializeStorageCipher(SharedPreferences configSource, final SecurePreferencesCallback<Void> callback) {
        try {
            this.storageCipherFactory = new StorageCipherFactory(configSource, this.config.getPrefOptionKeyCipherAlgorithm(), this.config.getPrefOptionStorageCipherAlgorithm(), this.config);
            if (this.storageCipherFactory.requiresReEncryption()) {
                Log.w((String)TAG, (String)"Algorithm changed detected.");
                this.handleKeyMismatch(configSource, callback, null, "Algorithm changed detected");
                return;
            }
            Cipher cipher = this.storageCipherFactory.getCurrentKeyCipher(this.context).getCipher(this.context);
            boolean enforceRequired = this.config.getEnforceBiometrics();
            boolean deviceHasSecurity = this.isDeviceSecure();
            if (cipher == null || Build.VERSION.SDK_INT < 28 || !enforceRequired && !deviceHasSecurity) {
                this.storageCipher = this.storageCipherFactory.getCurrentStorageCipher(this.context, cipher);
                callback.onSuccess(null);
                return;
            }
            this.authenticateUser(cipher, new SecurePreferencesCallback<BiometricPrompt.AuthenticationResult>(){

                @Override
                public void onSuccess(BiometricPrompt.AuthenticationResult result) {
                    try {
                        FlutterSecureStorage.this.storageCipher = FlutterSecureStorage.this.storageCipherFactory.getCurrentStorageCipher(FlutterSecureStorage.this.context, result.getCryptoObject().getCipher());
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Biometric authentication succeeded");
                    }
                    catch (Exception e) {
                        Log.e((String)FlutterSecureStorage.TAG, (String)"Failed to initialize storage cipher after authentication", (Throwable)e);
                        callback.onError(e);
                    }
                    callback.onSuccess(null);
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        }
        catch (BadPaddingException e) {
            this.handleKeyMismatch(configSource, callback, e, "Bad padding, wrong key for cipher algorithm");
        }
        catch (InvalidKeyException e) {
            this.handleKeyMismatch(configSource, callback, e, "Invalid key, key type incompatible with cipher");
        }
        catch (IllegalBlockSizeException e) {
            this.handleKeyMismatch(configSource, callback, e, "Illegal block size, wrong cipher configuration");
        }
        catch (NoSuchAlgorithmException e) {
            Log.e((String)TAG, (String)"Cryptographic algorithm not available on this device", (Throwable)e);
            callback.onError(new Exception("Required cryptographic algorithm not supported by device.", e));
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Failed to initialize storage cipher", (Throwable)e);
            callback.onError(e);
        }
    }

    private void migrateData(SharedPreferences configSource, SharedPreferences dataSource, SecurePreferencesCallback<Void> callback) {
        Log.i((String)TAG, (String)"Starting data migration from saved to current cipher algorithms...");
        try {
            String savedStorageAlg = this.storageCipherFactory.getSavedKeyCipher(this.context).toString();
            String currentStorageAlg = this.config.getPrefOptionStorageCipherAlgorithm();
            boolean fromBiometric = this.isBiometricAlgorithm(savedStorageAlg);
            boolean toBiometric = this.isBiometricAlgorithm(currentStorageAlg);
            if (fromBiometric || toBiometric) {
                Log.i((String)TAG, (String)("Detected biometric migration: FROM=" + savedStorageAlg + ", TO=" + currentStorageAlg));
                this.migrateBiometric(configSource, dataSource, fromBiometric, toBiometric, callback);
            } else {
                Log.i((String)TAG, (String)("Detected non-biometric migration: FROM=" + savedStorageAlg + ", TO=" + currentStorageAlg));
                this.migrateNonBiometric(configSource, dataSource, callback);
            }
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Failed to start migration", (Throwable)e);
            callback.onError(new Exception("Migration initialization failed", e));
        }
    }

    private Map<String, String> decryptAllWithSavedCipher(SharedPreferences dataSource, StorageCipher savedStorageCipher) throws Exception {
        HashMap<String, String> decryptedCache = new HashMap<String, String>();
        int count = 0;
        for (Map.Entry entry : dataSource.getAll().entrySet()) {
            String key = (String)entry.getKey();
            Object value = entry.getValue();
            if (!(value instanceof String) || !key.contains(this.config.getSharedPreferencesKeyPrefix())) continue;
            try {
                byte[] encryptedData = Base64.decode((String)((String)value), (int)0);
                byte[] decryptedData = savedStorageCipher.decrypt(encryptedData);
                String plainValue = new String(decryptedData, charset);
                decryptedCache.put(key, plainValue);
                ++count;
            }
            catch (Exception e) {
                Log.e((String)TAG, (String)("Failed to decrypt key: " + key), (Throwable)e);
                throw new Exception("Failed to decrypt existing data with saved cipher for key: " + key, e);
            }
        }
        Log.d((String)TAG, (String)("Successfully decrypted " + count + " items with saved cipher"));
        return decryptedCache;
    }

    private void encryptAllWithCurrentCipher(Map<String, String> cache, SharedPreferences dataTarget, StorageCipher currentStorageCipher) throws Exception {
        SharedPreferences.Editor editor = dataTarget.edit();
        int count = 0;
        for (Map.Entry<String, String> entry : cache.entrySet()) {
            try {
                byte[] encryptedData = currentStorageCipher.encrypt(entry.getValue().getBytes(charset));
                String encodedValue = Base64.encodeToString((byte[])encryptedData, (int)0);
                editor.putString(entry.getKey(), encodedValue);
                ++count;
            }
            catch (Exception e) {
                Log.e((String)TAG, (String)("Failed to encrypt key: " + entry.getKey()), (Throwable)e);
                throw new Exception("Failed to encrypt data with current cipher for key: " + entry.getKey(), e);
            }
        }
        editor.apply();
        Log.d((String)TAG, (String)("Successfully encrypted and saved " + count + " items with current cipher"));
    }

    private boolean isBiometricAlgorithm(String algorithmName) {
        return algorithmName != null && algorithmName.contains("BIOMETRIC");
    }

    private void migrateNonBiometric(SharedPreferences configSource, SharedPreferences dataSource, SecurePreferencesCallback<Void> callback) {
        Log.i((String)TAG, (String)"Starting non-biometric migration (no authentication required)...");
        try {
            Log.d((String)TAG, (String)"Step 1/6: Initializing saved cipher...");
            StorageCipher savedCipher = this.storageCipherFactory.getSavedStorageCipher(this.context, null);
            Log.d((String)TAG, (String)"Step 2/6: Decrypting all data with saved cipher...");
            Map<String, String> decryptedCache = this.decryptAllWithSavedCipher(dataSource, savedCipher);
            Log.d((String)TAG, (String)"Step 3/6: Deleting old RSA key from Android KeyStore...");
            if (this.storageCipherFactory.changedKeyAlgorithm()) {
                try {
                    KeyCipher savedKeyCipher = this.storageCipherFactory.getSavedKeyCipher(this.context);
                    savedKeyCipher.deleteKey();
                    savedCipher.deleteKey(this.context);
                    Log.d((String)TAG, (String)"Old key deleted from KeyStore");
                }
                catch (Exception deleteError) {
                    Log.w((String)TAG, (String)"Failed to delete old key from KeyStore (may not exist)", (Throwable)deleteError);
                }
            }
            Log.d((String)TAG, (String)"Step 4/6: Updating algorithm markers to current...");
            this.updateAlgorithmMarkers(configSource);
            Log.d((String)TAG, (String)"Step 5/6: Initializing current cipher with fresh AES key...");
            StorageCipher currentCipher = this.storageCipherFactory.getCurrentStorageCipher(this.context, null);
            if (decryptedCache.isEmpty()) {
                Log.i((String)TAG, (String)"Step 6/6: No data to migrate, continuing...");
            } else {
                Log.d((String)TAG, (String)"Step 6/6: Encrypting all data with current cipher...");
                this.encryptAllWithCurrentCipher(decryptedCache, dataSource, currentCipher);
            }
            this.storageCipher = currentCipher;
            Log.i((String)TAG, (String)("Non-biometric migration completed successfully! Migrated " + decryptedCache.size() + " items."));
            callback.onSuccess(null);
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Non-biometric migration failed", (Throwable)e);
            callback.onError(new Exception("Non-biometric migration failed", e));
        }
    }

    private void updateAlgorithmMarkers(SharedPreferences configSource) {
        SharedPreferences.Editor editor = configSource.edit();
        this.storageCipherFactory.storeCurrentAlgorithms(editor);
        editor.commit();
        Log.d((String)TAG, (String)"Algorithm markers updated to current");
    }

    private void migrateBiometric(SharedPreferences configSource, SharedPreferences dataSource, boolean fromBiometric, boolean toBiometric, SecurePreferencesCallback<Void> callback) {
        Log.i((String)TAG, (String)"Starting biometric migration (authentication required)...");
        Log.i((String)TAG, (String)("Migration direction: FROM biometric=" + fromBiometric + ", TO biometric=" + toBiometric));
        try {
            if (fromBiometric && !toBiometric) {
                Log.i((String)TAG, (String)"You will be prompted to authenticate with your OLD biometric settings to decrypt existing data.");
                this.migrateFromBiometricToNonBiometric(configSource, dataSource, callback);
            } else if (!fromBiometric && toBiometric) {
                Log.i((String)TAG, (String)"You will be prompted to authenticate with your NEW biometric settings to encrypt data.");
                this.migrateFromNonBiometricToBiometric(configSource, dataSource, callback);
            } else {
                Log.i((String)TAG, (String)"You will be prompted to authenticate twice (once for decrypt, once for encrypt).");
                this.migrateBiometricToBiometric(configSource, dataSource, callback);
            }
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Biometric migration failed", (Throwable)e);
            callback.onError(new Exception("Biometric migration failed", e));
        }
    }

    private void migrateFromBiometricToNonBiometric(final SharedPreferences configSource, final SharedPreferences dataSource, final SecurePreferencesCallback<Void> callback) {
        try {
            Log.d((String)TAG, (String)"Step 1/6: Getting saved biometric cipher...");
            KeyCipher savedKeyCipher = this.storageCipherFactory.getSavedKeyCipher(this.context);
            final Cipher oldKeyCipher = savedKeyCipher.getCipher(this.context);
            if (oldKeyCipher == null) {
                throw new Exception("Failed to get saved biometric cipher");
            }
            Log.i((String)TAG, (String)"Authenticating with OLD biometric cipher to decrypt data...");
            this.authenticateUser(oldKeyCipher, new SecurePreferencesCallback<BiometricPrompt.AuthenticationResult>(){

                @Override
                public void onSuccess(BiometricPrompt.AuthenticationResult unused) {
                    try {
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 2/6: Decrypting all data with saved biometric cipher...");
                        StorageCipher savedCipher = FlutterSecureStorage.this.storageCipherFactory.getSavedStorageCipher(FlutterSecureStorage.this.context, oldKeyCipher);
                        Map<String, String> decryptedCache = FlutterSecureStorage.this.decryptAllWithSavedCipher(dataSource, savedCipher);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 3/6: Deleting old biometric AES key from Android KeyStore...");
                        if (FlutterSecureStorage.this.storageCipherFactory.changedKeyAlgorithm()) {
                            try {
                                KeyCipher savedKeyCipher = FlutterSecureStorage.this.storageCipherFactory.getSavedKeyCipher(FlutterSecureStorage.this.context);
                                savedKeyCipher.deleteKey();
                                savedCipher.deleteKey(FlutterSecureStorage.this.context);
                                Log.d((String)FlutterSecureStorage.TAG, (String)"Old key deleted from KeyStore");
                            }
                            catch (Exception deleteError) {
                                Log.w((String)FlutterSecureStorage.TAG, (String)"Failed to delete old key from KeyStore (may not exist)", (Throwable)deleteError);
                            }
                        }
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 4/6: Updating algorithm markers to current...");
                        FlutterSecureStorage.this.updateAlgorithmMarkers(configSource);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 5/6: Initializing current non-biometric cipher...");
                        StorageCipher currentCipher = FlutterSecureStorage.this.storageCipherFactory.getCurrentStorageCipher(FlutterSecureStorage.this.context, null);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 6/6: Encrypting all data with current cipher...");
                        FlutterSecureStorage.this.encryptAllWithCurrentCipher(decryptedCache, dataSource, currentCipher);
                        FlutterSecureStorage.this.storageCipher = currentCipher;
                        Log.i((String)FlutterSecureStorage.TAG, (String)"Biometric\u2192Non-biometric migration completed! Data no longer requires biometric authentication.");
                        callback.onSuccess(null);
                    }
                    catch (Exception e) {
                        Log.e((String)FlutterSecureStorage.TAG, (String)"Failed to complete migration after authentication", (Throwable)e);
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    Log.e((String)FlutterSecureStorage.TAG, (String)"Biometric authentication failed for migration", (Throwable)e);
                    callback.onError(new Exception("Migration cancelled: Biometric authentication failed", e));
                }
            });
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Failed to initialize biometric migration", (Throwable)e);
            callback.onError(e);
        }
    }

    private void migrateFromNonBiometricToBiometric(SharedPreferences configSource, final SharedPreferences dataSource, final SecurePreferencesCallback<Void> callback) {
        try {
            Log.d((String)TAG, (String)"Step 1/6: Decrypting all data with saved non-biometric cipher...");
            StorageCipher savedCipher = this.storageCipherFactory.getSavedStorageCipher(this.context, null);
            Map<String, String> decryptedCache = this.decryptAllWithSavedCipher(dataSource, savedCipher);
            Log.d((String)TAG, (String)"Step 2/6: Deleting old RSA key from Android KeyStore...");
            if (this.storageCipherFactory.changedKeyAlgorithm()) {
                try {
                    KeyCipher savedKeyCipher = this.storageCipherFactory.getSavedKeyCipher(this.context);
                    savedKeyCipher.deleteKey();
                    savedCipher.deleteKey(this.context);
                    Log.d((String)TAG, (String)"Old key deleted from KeyStore");
                }
                catch (Exception deleteError) {
                    Log.w((String)TAG, (String)"Failed to delete old key from KeyStore (may not exist)", (Throwable)deleteError);
                }
            }
            Log.d((String)TAG, (String)"Step 3/6: Updating algorithm markers to current...");
            this.updateAlgorithmMarkers(configSource);
            Log.d((String)TAG, (String)"Step 4/6: Getting current biometric cipher...");
            KeyCipher currentKeyCipher = this.storageCipherFactory.getCurrentKeyCipher(this.context);
            final Cipher newCipher = currentKeyCipher.getCipher(this.context);
            if (newCipher == null) {
                throw new Exception("Failed to get current biometric cipher");
            }
            Log.i((String)TAG, (String)"Authenticating with NEW biometric cipher to encrypt data...");
            final Map<String, String> cachedData = decryptedCache;
            this.authenticateUser(newCipher, new SecurePreferencesCallback<BiometricPrompt.AuthenticationResult>(){

                @Override
                public void onSuccess(BiometricPrompt.AuthenticationResult unused) {
                    try {
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 5/6: Initializing current biometric cipher...");
                        StorageCipher currentCipher = FlutterSecureStorage.this.storageCipherFactory.getCurrentStorageCipher(FlutterSecureStorage.this.context, newCipher);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 6/6: Encrypting all data with current biometric cipher...");
                        FlutterSecureStorage.this.encryptAllWithCurrentCipher(cachedData, dataSource, currentCipher);
                        FlutterSecureStorage.this.storageCipher = currentCipher;
                        Log.i((String)FlutterSecureStorage.TAG, (String)"Non-biometric\u2192Biometric migration completed! Data now requires biometric authentication.");
                        callback.onSuccess(null);
                    }
                    catch (Exception e) {
                        Log.e((String)FlutterSecureStorage.TAG, (String)"Failed to complete migration after authentication", (Throwable)e);
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    Log.e((String)FlutterSecureStorage.TAG, (String)"Biometric authentication failed for migration", (Throwable)e);
                    callback.onError(new Exception("Migration cancelled: Biometric authentication failed", e));
                }
            });
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Failed to initialize biometric migration", (Throwable)e);
            callback.onError(e);
        }
    }

    private void migrateBiometricToBiometric(final SharedPreferences configSource, final SharedPreferences dataSource, final SecurePreferencesCallback<Void> callback) {
        try {
            Log.d((String)TAG, (String)"Step 1/7: Getting saved biometric cipher...");
            KeyCipher savedKeyCipher = this.storageCipherFactory.getSavedKeyCipher(this.context);
            final Cipher oldCipher = savedKeyCipher.getCipher(this.context);
            if (oldCipher == null) {
                throw new Exception("Failed to get saved biometric cipher");
            }
            Log.i((String)TAG, (String)"Authenticating with OLD biometric cipher to decrypt data...");
            this.authenticateUser(oldCipher, new SecurePreferencesCallback<BiometricPrompt.AuthenticationResult>(){

                @Override
                public void onSuccess(BiometricPrompt.AuthenticationResult unused) {
                    try {
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 2/7: Decrypting all data with saved biometric cipher...");
                        StorageCipher savedCipher = FlutterSecureStorage.this.storageCipherFactory.getSavedStorageCipher(FlutterSecureStorage.this.context, oldCipher);
                        Map<String, String> decryptedCache = FlutterSecureStorage.this.decryptAllWithSavedCipher(dataSource, savedCipher);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 3/7: Deleting old biometric AES key from Android KeyStore...");
                        if (FlutterSecureStorage.this.storageCipherFactory.changedKeyAlgorithm()) {
                            try {
                                KeyCipher savedKeyCipher = FlutterSecureStorage.this.storageCipherFactory.getSavedKeyCipher(FlutterSecureStorage.this.context);
                                savedKeyCipher.deleteKey();
                                savedCipher.deleteKey(FlutterSecureStorage.this.context);
                                Log.d((String)FlutterSecureStorage.TAG, (String)"Old key deleted from KeyStore");
                            }
                            catch (Exception deleteError) {
                                Log.w((String)FlutterSecureStorage.TAG, (String)"Failed to delete old key from KeyStore (may not exist)", (Throwable)deleteError);
                            }
                        }
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 4/7: Updating algorithm markers to current...");
                        FlutterSecureStorage.this.updateAlgorithmMarkers(configSource);
                        Log.d((String)FlutterSecureStorage.TAG, (String)"Step 5/7: Getting current biometric cipher...");
                        KeyCipher currentKeyCipher = FlutterSecureStorage.this.storageCipherFactory.getCurrentKeyCipher(FlutterSecureStorage.this.context);
                        final Cipher newCipher = currentKeyCipher.getCipher(FlutterSecureStorage.this.context);
                        if (newCipher == null) {
                            throw new Exception("Failed to get current biometric cipher");
                        }
                        Log.i((String)FlutterSecureStorage.TAG, (String)"Authenticating with NEW biometric cipher to encrypt data...");
                        final Map<String, String> cachedData = decryptedCache;
                        FlutterSecureStorage.this.authenticateUser(newCipher, new SecurePreferencesCallback<BiometricPrompt.AuthenticationResult>(){

                            @Override
                            public void onSuccess(BiometricPrompt.AuthenticationResult unused) {
                                try {
                                    Log.d((String)FlutterSecureStorage.TAG, (String)"Step 6/7: Initializing current biometric cipher...");
                                    StorageCipher currentCipher = FlutterSecureStorage.this.storageCipherFactory.getCurrentStorageCipher(FlutterSecureStorage.this.context, newCipher);
                                    Log.d((String)FlutterSecureStorage.TAG, (String)"Step 7/7: Encrypting all data with current biometric cipher...");
                                    FlutterSecureStorage.this.encryptAllWithCurrentCipher(cachedData, dataSource, currentCipher);
                                    FlutterSecureStorage.this.storageCipher = currentCipher;
                                    Log.i((String)FlutterSecureStorage.TAG, (String)"Biometric\u2192Biometric migration completed! Data now uses new biometric cipher.");
                                    callback.onSuccess(null);
                                }
                                catch (Exception e) {
                                    Log.e((String)FlutterSecureStorage.TAG, (String)"Failed to complete migration after second authentication", (Throwable)e);
                                    callback.onError(e);
                                }
                            }

                            @Override
                            public void onError(Exception e) {
                                Log.e((String)FlutterSecureStorage.TAG, (String)"Second biometric authentication failed for migration", (Throwable)e);
                                callback.onError(new Exception("Migration cancelled: Second biometric authentication failed", e));
                            }
                        });
                    }
                    catch (Exception e) {
                        Log.e((String)FlutterSecureStorage.TAG, (String)"Failed after first authentication", (Throwable)e);
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    Log.e((String)FlutterSecureStorage.TAG, (String)"First biometric authentication failed for migration", (Throwable)e);
                    callback.onError(new Exception("Migration cancelled: First biometric authentication failed", e));
                }
            });
        }
        catch (Exception e) {
            Log.e((String)TAG, (String)"Failed to initialize biometric-to-biometric migration", (Throwable)e);
            callback.onError(e);
        }
    }

    private void setEncryptedPrefsMigrated(SharedPreferences configSource) {
        SharedPreferences.Editor editor = configSource.edit();
        editor.putBoolean("ENCRYPTED_PREFERENCES_MIGRATED", true);
        editor.commit();
    }

    private Boolean getEncryptedPrefsMigrated(SharedPreferences configSource) {
        return configSource.getBoolean("ENCRYPTED_PREFERENCES_MIGRATED", false);
    }

    private void handleKeyMismatch(final SharedPreferences configSource, final SecurePreferencesCallback<Void> callback, Exception exception, final String errorType) {
        Log.e((String)TAG, (String)("Key mismatch detected during cipher initialization: " + errorType), (Throwable)exception);
        Log.e((String)TAG, (String)"This typically occurs after an algorithm change.");
        Log.e((String)TAG, (String)"Stored key cannot be decrypted with current algorithm.");
        if (this.config.shouldMigrateOnAlgorithmChange()) {
            Log.i((String)TAG, (String)"migrateOnAlgorithmChange is enabled. Attempting data migration...");
            SharedPreferences dataPrefs = this.context.getSharedPreferences(this.config.getSharedPreferencesName(), 0);
            this.migrateData(configSource, dataPrefs, new SecurePreferencesCallback<Void>(){

                @Override
                public void onSuccess(Void unused) {
                    Log.i((String)FlutterSecureStorage.TAG, (String)"Data migration completed successfully!");
                    FlutterSecureStorage.this.setEncryptedPrefsMigrated(configSource);
                    callback.onSuccess(null);
                }

                @Override
                public void onError(Exception migrationError) {
                    Log.e((String)FlutterSecureStorage.TAG, (String)("Data migration failed: " + migrationError.getMessage()), (Throwable)migrationError);
                    if (FlutterSecureStorage.this.config.shouldDeleteOnFailure()) {
                        Log.w((String)FlutterSecureStorage.TAG, (String)"resetOnError is enabled. Deleting all data as fallback...");
                        FlutterSecureStorage.this.deleteAllDataAndKeys(configSource, callback);
                        FlutterSecureStorage.this.setEncryptedPrefsMigrated(configSource);
                    } else {
                        Log.e((String)FlutterSecureStorage.TAG, (String)"Set resetOnError=true to automatically delete data after migration failure.");
                        String userMessage = String.format("Migration failed after algorithm change (%s). Enable resetOnError=true or call deleteAll().", errorType);
                        callback.onError(new Exception(userMessage, migrationError));
                    }
                }
            });
        } else {
            Log.w((String)TAG, (String)"migrateOnAlgorithmChange is disabled. Skipping data migration.");
            if (this.config.shouldDeleteOnFailure()) {
                Log.w((String)TAG, (String)"resetOnError is enabled. Deleting all data and keys to recover.");
                this.deleteAllDataAndKeys(configSource, callback);
            } else {
                Log.e((String)TAG, (String)"Set resetOnError=true to automatically delete data and recover.");
                Log.e((String)TAG, (String)"Or set migrateOnAlgorithmChange=true to preserve data during algorithm changes.");
                String userMessage = String.format("Key mismatch after algorithm change (%s). Enable migrateOnAlgorithmChange=true to preserve data, or resetOnError=true to delete.", errorType);
                callback.onError(new Exception(userMessage, exception));
            }
        }
    }

    private void deleteAllDataAndKeys(SharedPreferences configSource, SecurePreferencesCallback<Void> callback) {
        try {
            try {
                KeyCipher cipher = this.storageCipherFactory.getCurrentKeyCipher(this.context);
                cipher.deleteKey();
                Log.i((String)TAG, (String)"Deleted key from AndroidKeyStore");
            }
            catch (Exception keyDeleteError) {
                Log.w((String)TAG, (String)"Failed to delete key from AndroidKeyStore (may not exist)", (Throwable)keyDeleteError);
            }
            SharedPreferences dataPrefs = this.context.getSharedPreferences(this.config.getSharedPreferencesName(), 0);
            dataPrefs.edit().clear().apply();
            Log.d((String)TAG, (String)"Deleted all encrypted data");
            SharedPreferences keyPrefs = this.context.getSharedPreferences("FlutterSecureKeyStorage", 0);
            keyPrefs.edit().clear().apply();
            Log.d((String)TAG, (String)"Deleted wrapped keys from SharedPreferences");
            SharedPreferences.Editor editor = configSource.edit();
            this.storageCipherFactory.storeCurrentAlgorithms(editor);
            editor.apply();
            Log.d((String)TAG, (String)"Updated algorithm markers to current");
            Log.w((String)TAG, (String)"All data deleted. Reinitializing with new algorithm...");
            this.initializeStorageCipher(configSource, callback);
        }
        catch (Exception cleanupError) {
            Log.e((String)TAG, (String)"Failed to clean up after key mismatch", (Throwable)cleanupError);
            callback.onError(cleanupError);
        }
    }

    public boolean isBiometricAvailable() {
        if (Build.VERSION.SDK_INT >= 30) {
            BiometricManager biometricManager = (BiometricManager)this.context.getSystemService(BiometricManager.class);
            if (biometricManager == null) {
                return false;
            }
            int result = biometricManager.canAuthenticate(32783);
            return result == 0 && this.isDeviceSecure();
        }
        return this.isDeviceSecure();
    }

    public boolean isDeviceSecure() {
        KeyguardManager keyguardManager = (KeyguardManager)this.context.getSystemService("keyguard");
        return keyguardManager != null && keyguardManager.isDeviceSecure();
    }

    private void ensureBiometricAvailable(boolean enforceRequired) throws Exception {
        if (Build.VERSION.SDK_INT < 28) {
            if (enforceRequired) {
                throw new Exception("BIOMETRIC_UNAVAILABLE: Biometric authentication requires Android 9 (API 28) or higher");
            }
            return;
        }
        if (!this.isDeviceSecure()) {
            if (enforceRequired) {
                throw new Exception("BIOMETRIC_UNAVAILABLE: Device has no PIN, pattern, password, or biometric enrolled. Please secure your device in Settings.");
            }
            Log.w((String)TAG, (String)"Device has no security. Biometric authentication will be skipped (enforceBiometrics=false).");
            return;
        }
        if (Build.VERSION.SDK_INT >= 30) {
            BiometricManager biometricManager = (BiometricManager)this.context.getSystemService(BiometricManager.class);
            if (biometricManager == null) {
                if (enforceRequired) {
                    throw new Exception("BIOMETRIC_UNAVAILABLE: BiometricManager not available on this device");
                }
                return;
            }
            int result = biometricManager.canAuthenticate(32783);
            switch (result) {
                case 0: {
                    return;
                }
                case 12: {
                    if (!enforceRequired) break;
                    throw new Exception("BIOMETRIC_UNAVAILABLE: No biometric hardware detected on this device");
                }
                case 1: {
                    if (!enforceRequired) break;
                    throw new Exception("BIOMETRIC_UNAVAILABLE: Biometric hardware temporarily unavailable");
                }
                case 11: {
                    if (!enforceRequired) break;
                    throw new Exception("BIOMETRIC_UNAVAILABLE: No fingerprint or face enrolled. Please enroll in Settings.");
                }
                case 15: {
                    if (!enforceRequired) break;
                    throw new Exception("BIOMETRIC_UNAVAILABLE: Security update required for biometric authentication");
                }
                default: {
                    if (!enforceRequired) break;
                    throw new Exception("BIOMETRIC_UNAVAILABLE: Unknown biometric status (code: " + result + ")");
                }
            }
            Log.w((String)TAG, (String)("Biometric check failed with code " + result + ", but continuing (enforceBiometrics=false)"));
        }
    }

    private void authenticateUser(Cipher cipher, final SecurePreferencesCallback<BiometricPrompt.AuthenticationResult> securePreferencesCallback) throws Exception {
        boolean enforceRequired = this.config.getEnforceBiometrics();
        this.ensureBiometricAvailable(enforceRequired);
        if (Build.VERSION.SDK_INT < 28) {
            if (enforceRequired) {
                throw new Exception("BIOMETRIC_UNAVAILABLE: Biometric authentication requires Android 9 (API 28) or higher");
            }
            return;
        }
        BiometricPrompt.CryptoObject crypto = new BiometricPrompt.CryptoObject(cipher);
        BiometricPrompt.Builder promptInfoBuilder = new BiometricPrompt.Builder(this.context).setTitle((CharSequence)this.config.getBiometricPromptTitle()).setSubtitle((CharSequence)this.config.getPrefOptionBiometricPromptSubtitle());
        if (Build.VERSION.SDK_INT >= 30) {
            promptInfoBuilder.setAllowedAuthenticators(32783);
        }
        BiometricPrompt promptInfo = promptInfoBuilder.build();
        CancellationSignal cancellationSignal = new CancellationSignal();
        ExecutorService executor = Executors.newSingleThreadExecutor();
        BiometricPrompt.AuthenticationCallback callback = new BiometricPrompt.AuthenticationCallback(){

            public void onAuthenticationSucceeded(BiometricPrompt.AuthenticationResult result) {
                super.onAuthenticationSucceeded(result);
                securePreferencesCallback.onSuccess(result);
            }

            public void onAuthenticationFailed() {
                super.onAuthenticationFailed();
                Log.w((String)FlutterSecureStorage.TAG, (String)"Biometric authentication failed, user not recognized");
                securePreferencesCallback.onError(new Exception("Biometric authentication failed, user not recognized"));
            }

            public void onAuthenticationError(int errorCode, CharSequence errString) {
                super.onAuthenticationError(errorCode, errString);
                Log.e((String)FlutterSecureStorage.TAG, (String)("Biometric authentication error [" + errorCode + "]: " + errString));
                securePreferencesCallback.onError(new Exception("Biometric authentication error [" + errorCode + "]: " + errString));
            }
        };
        promptInfo.authenticate(crypto, cancellationSignal, (Executor)executor, callback);
    }

    private boolean hasDataInEncryptedSharedPreferences(SharedPreferences encryptedPreferences) {
        Map all = encryptedPreferences.getAll();
        for (String key : all.keySet()) {
            if (!key.contains(this.config.getSharedPreferencesKeyPrefix())) continue;
            return true;
        }
        return false;
    }

    private void migrateFromEncryptedSharedPreferences(SharedPreferences source, SharedPreferences target) throws Exception {
        int migratedCount = 0;
        for (Map.Entry entry : source.getAll().entrySet()) {
            Object v = entry.getValue();
            String key = (String)entry.getKey();
            if (!(v instanceof String)) continue;
            String plainValue = (String)v;
            if (!key.contains(this.config.getSharedPreferencesKeyPrefix())) continue;
            byte[] encrypted = this.storageCipher.encrypt(plainValue.getBytes(charset));
            String baseEncoded = Base64.encodeToString((byte[])encrypted, (int)0);
            target.edit().putString(key, baseEncoded).apply();
            source.edit().remove(key).apply();
            ++migratedCount;
            Log.d((String)TAG, (String)("Migrated key: " + key.replaceFirst(this.config.getSharedPreferencesKeyPrefix() + '_', "")));
        }
        Log.i((String)TAG, (String)("Migration complete: " + migratedCount + " items migrated from EncryptedSharedPreferences to custom cipher storage"));
    }

    private SharedPreferences initializeEncryptedSharedPreferencesManager(Context context) throws GeneralSecurityException, IOException {
        MasterKey key = new MasterKey.Builder(context).setKeyGenParameterSpec(new KeyGenParameterSpec.Builder("_androidx_security_master_key_", 3).setEncryptionPaddings(new String[]{"NoPadding"}).setBlockModes(new String[]{"GCM"}).setKeySize(256).build()).build();
        return EncryptedSharedPreferences.create(context, this.config.getSharedPreferencesName(), key, EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV, EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM);
    }

    private boolean handleStorageError(String operation, String key, Exception error) {
        boolean deleteOnFailure = this.config.shouldDeleteOnFailure();
        String target = key != null ? "key '" + key + "'" : "all data";
        Log.e((String)TAG, (String)String.format("Storage operation '%s' failed for %s. %s", operation, target, deleteOnFailure ? "Attempting to delete corrupted data and retry..." : "Set resetOnError=true to automatically delete corrupted data."), (Throwable)error);
        if (!deleteOnFailure) {
            return false;
        }
        try {
            if (key != null) {
                this.delete(key);
            } else {
                this.deleteAll();
            }
            Log.w((String)TAG, (String)String.format("%s completed. Retrying operation...", key != null ? "Data for key has been deleted" : "All data has been deleted"));
            return true;
        }
        catch (Exception deleteError) {
            Log.e((String)TAG, (String)String.format("Failed to %s during error handling.", key != null ? "delete data for key" : "delete all data"), (Throwable)deleteError);
            return false;
        }
    }

    private String decodeRawValue(String value) throws Exception {
        if (value == null) {
            return null;
        }
        byte[] data = Base64.decode((String)value, (int)0);
        byte[] result = this.storageCipher.decrypt(data);
        return new String(result, charset);
    }
}

